function [dyn, u] = driverModelPredictive(dyn, mission, vehModel, veh, u, driver)
% vehAcc: current acceleration
% vehSpd: current speed
% vehSpd_next: speed at the next timestep

vehSpd = dyn.vehSpd;
vehSpd_next = dyn.vehSpd + dyn.vehAcc * veh.dt;
vehAcc = dyn.vehAcc;

GN = u{1};

%% Determine the driver state
% Lookahead braking/coasting
previewDist = 10 * (vehSpd*3.6);

vehSpd = min(vehSpd,0);
stopDist = veh.ctrl.accLimitLo.accelDist(vehSpd, 0);

previewDist = max([2*veh.dx, previewDist, stopDist]);
previewDist = min(previewDist, mission.finalDist - dyn.vehDist);

% By default
driver.state = "drive";

% Evaluate stop distance
% targetDist is a vector of preview distances to be evaluated
targetDist = veh.dx:veh.dx:previewDist;

% Check if there is an upcoming stop within the preview distance
stopIdx = find(dyn.vehDist < mission.stops.distance ...
    & mission.stops.distance <= ( dyn.vehDist + previewDist ), ...
    1, 'first');
% If there is, do not look beyond the stop.
if ~isempty(stopIdx)
    targetDist = sort([targetDist, mission.stops.distance(stopIdx) - dyn.vehDist]);
    targetDist( targetDist > (mission.stops.distance(stopIdx) - dyn.vehDist) ) = [];
end
targetSpd = mission.speed_kmh( dyn.vehDist + targetDist ) / 3.6;

brakeDist = veh.ctrl.accLimitLo.accelDist(vehSpd, targetSpd);
brakeDist(targetSpd > vehSpd) = 0;

if any(brakeDist > targetDist)
    driver.state = "brake";
end

% % Coasting
% if vehSpd > driver.coasting.vehSpdMin
% 
%     potEnergyDiff = veh.body.mass * 9.81 * previewDist * ( dyn.vehGrad / 100 );
%     targetSpd = mission.speed_kmh(dyn.vehDist + previewDist) ./ 3.6;
%     kinEnergyDiff = 0.5 * veh.body.mass * ( vehSpd^2 -  targetSpd.^2 );
%     energyDiff = kinEnergyDiff + potEnergyDiff;
% 
%     avgForce = energyDiff / previewDist;
% 
%     vehSlope = atan(dyn.vehGrad/100); % rad
%     [~, ~, vehPrf] = vehDynamicsModel(vehSpd, 0, vehSlope, veh.body, veh.wh);
%     resForce = vehPrf.vehForce;
% 
%     % Decision factor
%     DF_targetSpd = driver.coasting.targetSpdDecisionFactor(targetSpd);
%     DF_spdDrop = driver.coasting.spdDropDecisionFactor( vehSpd - targetSpd );
%     DF = 2.5 - 1.5 * DF_targetSpd * DF_spdDrop;
% 
%     if resForce * DF < avgForce
%         % Trigger coasting
%         driver.state = "coast";
%     else
%         % Brake?
%         driver.state = "brake";
%     end
% end

%% Behavioural acc. limits
if isfield(veh.ctrl, 'accLimitUp')
    accLimitUp = veh.ctrl.accLimitUp.eval(vehSpd);
end
if isfield(veh.ctrl, 'accLimitLo')
    accLimitLo = veh.ctrl.accLimitLo.eval(vehSpd);
end

if vehAcc > accLimitUp
    vehAcc = accLimitUp;
    vehSpd_next = vehSpd + vehAcc .* veh.dt;
elseif vehAcc < accLimitLo
    vehAcc = accLimitLo;
    vehSpd_next = vehSpd + vehAcc .* veh.dt;
end

%% Physical limits
switch driver.state
    case "drive"
        % Try the desired acceleration
        GN = max( GN, 1 );
        brakeCmd = 0;
        u = {GN, brakeCmd};
        dyn.vehAcc = vehAcc;
        [~, unfeas] = vehModel(u, dyn);

        if unfeas
            % Search maximum acceleration
            vehAccGrid = -1:0.01:vehAcc; % m/s^2
            try
                dyn.vehAcc = searchAcc(vehAccGrid, "max");
            catch
                driver.state = "brake";

                dyn.vehAcc = vehAcc;
                brakeCmdGrid = 0:0.01:1;

                brakeCmd = searchbrakeCmd(brakeCmdGrid);

            end
        end

    case "coast"
        % Search minimum acceleration without mechanical brakes
        vehAccGrid = -2:-0.01:vehAcc; % m/s^2
        try
            vehAcc = searchAcc(vehAccGrid, "min");
        end
        brakeCmd = 0;

    case "brake"
        % Set vehAcc to the behavioral decel curve
        dyn.vehAcc = veh.ctrl.accLimitLo.eval(dyn.vehSpd);
        % Find the brakeCmd that provides this acceleration
        brakeCmdGrid = 0:0.01:1;
        brakeCmd = searchbrakeCmd(brakeCmdGrid);
        
end

%% Limits based on engine and e-machines speed limits
if vehSpd_next > veh.maxSpd
    vehSpd_next = veh.maxSpd;
    dyn.vehAcc = ( vehSpd_next - vehSpd ) / veh.dt;
end

%% Return new brake command
u{2} = brakeCmd;

%% Helper functions
    function vehAcc = searchAcc(vehAccGrid, minOrMax)
        vehAccGrid = sort(vehAccGrid);

        % Try the provided acceleration grid
        dyn.vehAcc = vehAccGrid;
        [~, unfeas] = vehModel(u, dyn);

        switch minOrMax
            case "min"
                % Index of the strongest deceleration that is feasible
                maxAccIdx = find(unfeas == 0, 1, 'first');
            case "max"
                % Index of the strongest acceleration that is feasible
                maxAccIdx = find(unfeas == 0, 1, 'last');
        end

        if ~isempty(maxAccIdx)
            % Keep the strongest acceleration/deceleration that is feasible and
            % re-evaluate vehSpd_next
            vehAcc = vehAccGrid(maxAccIdx);
        end
    end

    function brakeCmd = searchbrakeCmd(brakeCmdGrid)

        brakeCmdGrid = sort(brakeCmdGrid);

        % Try the provided brake positions
        GN = max( GN, 1 );
        u = {GN, brakeCmdGrid};
        [~, unfeas] = vehModel(u, dyn);

        % Index of the smallest brakeCmd that gives the desired
        % deceleration/acceleration
        minBrakeIdx = find(unfeas == 0, 1, 'first');

        if ~isempty(minBrakeIdx)
            % Keep
            brakeCmd = brakeCmdGrid(minBrakeIdx);
        else
            brakeCmd = 1;
        end

    end


end